/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: BusStateDlg.cpp $
   Last Modification:
    $Author: MichaelT $
    $Modtime: 4.04.07 19:21 $
    $Revision: 1353 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
    Bus State Get/Set Test dialog
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
      1        04.04.2007  RM       initial version
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file BusStateDlg.cpp
///   Bus State Get/Set Test dialog
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "cifXTest.h"
#include "BusStateDlg.h"
#include "CifXTestDlg.h"

#define DEFAULT_SETBUSSTATE_TIMEOUT  0

/////////////////////////////////////////////////////////////////////////////
/// Bus state elements for list control
/////////////////////////////////////////////////////////////////////////////
static struct BUS_STATEtag
{
  LPCTSTR       szState;  //!< State to display in control
  unsigned long ulState;  //!< State value for element
} s_atBusState[] =
{
  {_T("Bus ON"),  CIFX_BUS_STATE_ON},
  {_T("Bus OFF"), CIFX_BUS_STATE_OFF},
};

IMPLEMENT_DYNAMIC(CBusStateDlg, CBaseDialog)

/////////////////////////////////////////////////////////////////////////////
/// Default Constructor
///   \param pParent Parent Window
/////////////////////////////////////////////////////////////////////////////
CBusStateDlg::CBusStateDlg(CWnd* pParent /*=NULL*/)
	: CBaseDialog(CBusStateDlg::IDD, pParent, false)
  , m_ulSetTimeout(DEFAULT_SETBUSSTATE_TIMEOUT)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CBusStateDlg::~CBusStateDlg()
{
}

/////////////////////////////////////////////////////////////////////////////
/// DDX/DDV support
///   \param pDX 
/////////////////////////////////////////////////////////////////////////////
void CBusStateDlg::DoDataExchange(CDataExchange* pDX)
{
  CBaseDialog::DoDataExchange(pDX);
  DDX_Control(pDX, IDC_CB_NEWSTATE, m_cNewStateCtrl);
}


BEGIN_MESSAGE_MAP(CBusStateDlg, CBaseDialog)
  ON_BN_CLICKED(IDC_BTN_GETSTATE, OnBnClickedBtnGetstate)
  ON_BN_CLICKED(IDC_BTN_SETSTATE, OnBnClickedBtnSetstate)
  ON_EN_KILLFOCUS(IDC_EDT_TIMEOUT, OnEnKillfocusEdtTimeout)
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
/// Button "Get Bus State" clicked event
/////////////////////////////////////////////////////////////////////////////
void CBusStateDlg::OnBnClickedBtnGetstate()
{
  uint32_t ulActState = 0;

  long lRet = CcifXTestDlg::s_pcDevice->BusState(CIFX_BUS_STATE_GETSTATE, &ulActState, m_ulSetTimeout);

  CString csError;
  if(CIFX_NO_ERROR != lRet)
  {
    csError.Format(_T("0x%08X\r\n%s"),
                   lRet,
                   (LPCTSTR)CcifXTestDlg::s_pcDevice->GetErrorDescription(lRet));

  } else
  {
    CString csState;

    switch(ulActState)
    {
    case CIFX_BUS_STATE_OFF:
      csState = _T("Bus OFF");
      break;

    case CIFX_BUS_STATE_ON:
      csState = _T("Bus ON");
      break;

    default:
      csState = _T("Illegal Bus State");
      break;
    }

    SetDlgItemText(IDC_EDT_ACTSTATE, csState);
  }
  SetDlgItemText(IDC_LASTERR, csError);

}

/////////////////////////////////////////////////////////////////////////////
/// Button "Set Bus State" clicked event
/////////////////////////////////////////////////////////////////////////////
void CBusStateDlg::OnBnClickedBtnSetstate()
{
  int iItem = m_cNewStateCtrl.GetCurSel();
  ASSERT(-1 != iItem);

  CWaitCursor cWait;

  uint32_t ulNewState = (uint32_t)m_cNewStateCtrl.GetItemData(iItem);
  uint32_t ulTemp     = 0;

  long lRet = CcifXTestDlg::s_pcDevice->BusState(ulNewState, &ulTemp, m_ulSetTimeout);

  CString csError;
  if(CIFX_NO_ERROR != lRet)
  {
    csError.Format(_T("0x%08X\r\n%s"),
                   lRet,
                   (LPCTSTR)CcifXTestDlg::s_pcDevice->GetErrorDescription(lRet));

  }
  SetDlgItemText(IDC_LASTERR, csError);
}

/////////////////////////////////////////////////////////////////////////////
/// Set Bus State Timeout editing end event
/////////////////////////////////////////////////////////////////////////////
void CBusStateDlg::OnEnKillfocusEdtTimeout()
{
  CString csTemp;
  GetDlgItemText(IDC_EDT_TIMEOUT, csTemp);

  TCHAR* pszEnd = NULL;
  m_ulSetTimeout = _tcstoul(csTemp, &pszEnd, 10);

  csTemp.Format(_T("%u"), m_ulSetTimeout);
  SetDlgItemText(IDC_EDT_TIMEOUT, csTemp);
}

/////////////////////////////////////////////////////////////////////////////
/// First Time dialog initialization
///   \return TRUE
/////////////////////////////////////////////////////////////////////////////
BOOL CBusStateDlg::OnInitDialog()
{
  CBaseDialog::OnInitDialog();

  SetDlgItemInt(IDC_EDT_TIMEOUT, m_ulSetTimeout, FALSE);

  for(int iIdx = 0; iIdx < sizeof(s_atBusState) / sizeof(s_atBusState[0]); ++iIdx)
  {
    int iItem = m_cNewStateCtrl.AddString(s_atBusState[iIdx].szState);
    m_cNewStateCtrl.SetItemData(iItem, s_atBusState[iIdx].ulState);
  }

  m_cNewStateCtrl.SetCurSel(0);

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}

/////////////////////////////////////////////////////////////////////////////
/// Cyclic Function (unused)
/////////////////////////////////////////////////////////////////////////////
void CBusStateDlg::ThreadFunction()
{
  //not needed, as we have no cyclic actions to perform
}

/////////////////////////////////////////////////////////////////////////////
/// Device changed in Main Dialog
///   \param pcDevice new Device (NULL if device was closed)
/////////////////////////////////////////////////////////////////////////////
void CBusStateDlg::OnUpdateDevice(CCifXDeviceBase* pcDevice)
{
  if(NULL != pcDevice)
    OnBnClickedBtnGetstate();
}
